﻿using gov.va.med.vbecs.BOL;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.ExceptionManagement;
using gov.va.med.vbecs.GUI;
using gov.va.med.vbecs.GUI.WPF;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Windows.Forms;
using WPF.PresentationLayer.Resources;
using WPF.PresentationLayer.UC115.Events;
using WPF.PresentationLayer.UC115.Views;

namespace WPF.PresentationLayer.UC115.ViewModels
{
    /// <summary>
    /// BloodUnitAboRhViewModel
    /// </summary>
    public class BloodUnitTabViewModel : PendingTestViewModelBase
    {
        #region Properties
        /// <summary>
        /// Command figured when row checkbox is checked
        /// </summary>
        public DelegateCommand<BloodUnitItemViewModel> ItemSelectedCommand { get; private set; }
        /// <summary>
        /// Command to open the product code selection dialog
        /// </summary>
        public DelegateCommand<BloodUnitItemViewModel> ProductCodeCommand { get; private set; }
        /// <summary>
        /// Command for filtering by unit id or product code
        /// </summary>
        public DelegateCommand SearchCommand { get; private set; }
        /// <summary>
        /// Command to clear filtering criteria 
        /// </summary>
        public DelegateCommand ClearCommand { get; private set; }
        /// <summary>
        /// AcceptCommand
        /// </summary>
        public DelegateCommand AcceptCommand { get; set; }
        /// <summary>
        /// RejectCommand 
        /// </summary>
        public DelegateCommand RejectCommand { get; set; }

        private string _unitIdSearchText;
        /// <summary>
        /// UnitIdSearchText
        /// </summary>
        public string UnitIdSearchText
        {
            get
            {
                return _unitIdSearchText;
            }
            set
            {
                _unitIdSearchText = value;
                RaisePropertyChanged(() => UnitIdSearchText);
            }
        }

        private string _productCodeSearchText;
        /// <summary>
        /// ProductCodeSearchText
        /// </summary>
        public string ProductCodeSearchText
        {
            get
            {
                return _productCodeSearchText;
            }
            set
            {
                _productCodeSearchText = value;
                RaisePropertyChanged(() => ProductCodeSearchText);
            }
        }

        /// <summary>
        /// Full list of pending blood units (used to filter/search)
        /// </summary>
        public ObservableCollection<BloodUnitItemViewModel> PendingBloodUnitFullList { get; private set; }

        private ObservableCollection<BloodUnitItemViewModel> _pendingBloodUnitList;
        /// <summary>
        /// PendingBloodUnitAboRhList - bound to DataGrid
        /// </summary>
        public ObservableCollection<BloodUnitItemViewModel> PendingBloodUnitList
        {
            get
            {
                return _pendingBloodUnitList;
            }
            set
            {
                _pendingBloodUnitList = value;
                RaisePropertyChanged(() => PendingBloodUnitList);
            }
        }
        #endregion

        /// <summary>
        /// BloodUnitAboRhViewModel constructor
        /// </summary>
        /// <param name="vbecsBaseForm"></param>
        /// <param name="pendingBloodUnitAboRhViewModel"></param>
        public BloodUnitTabViewModel(VbecsBaseForm vbecsBaseForm, IEnumerable<BloodUnitItemViewModel> pendingBloodUnitAboRhViewModel, StrRes.StringResourceItem helpFileTopic)
            : base(vbecsBaseForm, helpFileTopic)
        {
            // BR_115.11 - Order by Unit ID             
            PendingBloodUnitList = pendingBloodUnitAboRhViewModel.OrderBy(x => x.PendingBloodUnitTestModel.BloodUnitId).ToObservableCollection();
            PendingBloodUnitFullList = PendingBloodUnitList;            

            ItemSelectedCommand = new DelegateCommand<BloodUnitItemViewModel>(OnItemSelectedCommand);
            ProductCodeCommand = new DelegateCommand<BloodUnitItemViewModel>(OnProductCodeCommand);
            SearchCommand = new DelegateCommand(OnSearchCommand);
            ClearCommand = new DelegateCommand(OnClearCommand);
            AcceptCommand = new DelegateCommand(OnAcceptCommand);
            RejectCommand = new DelegateCommand(OnRejectCommand);
        }

        /// <summary>
        /// Reject the checked tests
        /// BR_115.05
        /// </summary>
        private void OnRejectCommand()
        {
            try
            {
                var selectedTestList = PendingBloodUnitList.Where(x => x.IsSelected);

                if (selectedTestList.Any(x => string.IsNullOrWhiteSpace(x.ReviewComment)))
                {
                    GuiMessenger.ShowMessageBox(StrRes.ValidMsg.UC115.ReviewCommentRequired());
                    return;
                }

                var dlgResult = GuiMessenger.ShowMessageBox(StrRes.ConfMsg.UC115.RejectTest());
                if (dlgResult == DialogResult.Yes)
                {
                    // Store the review comment in the model before saving/rejecting
                    foreach (var testToReject in selectedTestList)
                    {
                        testToReject.PendingTestModel.ReviewComment = testToReject.ReviewComment;
                    }

                    var pendingTestModelList = selectedTestList.Select(s => s.PendingBloodUnitTestModel).ToList();

                    // Make db call to reject tests                
                    PendingBloodUnitTest.RejectTests(pendingTestModelList);

                    // Remove rejected tests from the DataGrid and full list
                    for (int idx = 0; idx < PendingBloodUnitList.Count; idx++)
                    {
                        var pendingTest = PendingBloodUnitList[idx];

                        if (pendingTest.IsSelected)
                        {
                            var selectedUnit = pendingTest as BloodUnitItemViewModel;
                            if (selectedUnit != null && 
                                selectedUnit.BloodUnitModel != null)
                            {
                                VbecsBaseForm.ReleaseLocks(selectedUnit.BloodUnitModel.BloodUnitStatusGuid);
                            }

                            PendingBloodUnitList.Remove(pendingTest);
                            PendingBloodUnitFullList.Remove(pendingTest);
                            idx--;
                        }
                    }

                    base.UncheckAll(PendingBloodUnitList);
                    foreach (BloodUnitItemViewModel tmp in PendingBloodUnitList)
                    {
                        if (tmp.BloodUnitModel != null) VbecsBaseForm.ReleaseLocks(tmp.BloodUnitModel.BloodUnitStatusGuid);
                    }
                    // Raise property changed on the list so the tab will check if it should be disabled
                    // if not tests exist anymore
                    RaisePropertyChanged(() => PendingBloodUnitList);
                    RaisePropertyChanged(() => PendingBloodUnitFullList);

                    // Fire event that tests were reviewed after we remove them from the list
                    Messenger.Default.Send(new PendingTestReviewedEvent(pendingTestModelList, false));
                }
            }
            catch (RowVersionException err)
            {
                ExceptionManager.Publish(err);
                GuiMessenger.ShowMessageBox(VbecsBaseForm, StrRes.SysErrMsg.Common.DataWasNotSavedBecauseOfRowversionViolation());
            }
        }

        /// <summary>
        /// Accept the checked tests
        /// BR_115.07
        /// </summary>
        private void OnAcceptCommand()
        {
            try
            {
                var selectedTestList = PendingBloodUnitList.Where(x => x.IsSelected);

                var dlgResult = GuiMessenger.ShowMessageBox(StrRes.ConfMsg.UC115.AcceptTest());
                if (dlgResult == DialogResult.Yes)
                {
                    // Store the review comment and overrides in the model before saving/rejecting
                    foreach (var testToAccept in selectedTestList)
                    {
                        testToAccept.PendingTestModel.ReviewComment = testToAccept.ReviewComment;
                        testToAccept.PendingBloodUnitTestModel.BloodUnitModel = testToAccept.BloodUnitModel.Clone();
                        testToAccept.PendingTestModel.TestValidation = testToAccept.TestValidation.Clone();
                    }

                    var pendingTestModelList = selectedTestList.Select(s => s.PendingBloodUnitTestModel).ToList();

                    // Make db call to reject tests        
                    bool isWorkLoadsDefined = false;
                    PendingBloodUnitTest.AcceptTests(LogonUser.LogonUserDivisionCode, pendingTestModelList, ref isWorkLoadsDefined);

                    if (!isWorkLoadsDefined)
                    {
                        GuiMessenger.ShowMessageBox(StrRes.InfoMsg.UC015.BR1502_NoWorkloadDefined());
                    }

                    // Remove rejected tests from the DataGrid and full list
                    for (int idx = 0; idx < PendingBloodUnitList.Count; idx++)
                    {
                        var pendingTest = PendingBloodUnitList[idx];

                        if (pendingTest.IsSelected)
                        {
                            var selectedUnit = pendingTest as BloodUnitItemViewModel;
                            if (selectedUnit != null &&
                                selectedUnit.BloodUnitModel != null)
                            {
                                VbecsBaseForm.ReleaseLocks(selectedUnit.BloodUnitModel.BloodUnitStatusGuid);
                            }

                            PendingBloodUnitList.Remove(pendingTest);
                            PendingBloodUnitFullList.Remove(pendingTest);
                            idx--;
                        }
                    }

                    base.UncheckAll(PendingBloodUnitList);
                    foreach (BloodUnitItemViewModel tmp in PendingBloodUnitList)
                    {
                        if (tmp.BloodUnitModel != null) VbecsBaseForm.ReleaseLocks(tmp.BloodUnitModel.BloodUnitStatusGuid);
                    }
                    // Raise property changed on the list so the tab will check if it should be disabled
                    // if not tests exist anymore
                    RaisePropertyChanged(() => PendingBloodUnitList);
                    RaisePropertyChanged(() => PendingBloodUnitFullList);

                    // Fire event that tests were reviewed after we remove them from the list
                    Messenger.Default.Send(new PendingTestReviewedEvent(pendingTestModelList, false));
                }
            }
            catch (RowVersionException err)
            {
                ExceptionManager.Publish(err);
                GuiMessenger.ShowMessageBox(VbecsBaseForm, StrRes.SysErrMsg.Common.DataWasNotSavedBecauseOfRowversionViolation());
            }
        }

        /// <summary>
        /// Clear filters
        /// </summary>
        private void OnClearCommand()
        {
            UnitIdSearchText = string.Empty;
            ProductCodeSearchText = string.Empty;
            OnSearchCommand(); 
        }

        /// <summary>
        /// Filter data based on the specified unit id and product code
        /// BR_115.09 - The unit ID and product code are used to identify a specific unit from inventory.
        /// </summary>
        private void OnSearchCommand()
        {
            UncheckAll(PendingBloodUnitList);
            foreach (BloodUnitItemViewModel tmp in PendingBloodUnitList)
            {
                if (tmp.BloodUnitModel != null) VbecsBaseForm.ReleaseLocks(tmp.BloodUnitModel.BloodUnitStatusGuid);
            }
            IEnumerable<BloodUnitItemViewModel> filteredList;

            if (!string.IsNullOrWhiteSpace(UnitIdSearchText) &&
                !string.IsNullOrWhiteSpace(ProductCodeSearchText))
            {
                filteredList = PendingBloodUnitFullList.Where(x => x.PendingBloodUnitTestModel.BloodUnitId.IndexOf(UnitIdSearchText, StringComparison.InvariantCultureIgnoreCase) >= 0
                                                           && x.PendingBloodUnitTestModel.BloodUnitModel != null
                                                           && x.PendingBloodUnitTestModel.BloodUnitModel.ProductCode.IndexOf(ProductCodeSearchText, StringComparison.InvariantCultureIgnoreCase) >= 0);
            }
            else if (!string.IsNullOrWhiteSpace(UnitIdSearchText))
            {
                filteredList = PendingBloodUnitFullList.Where(x => x.PendingBloodUnitTestModel.BloodUnitId.IndexOf(UnitIdSearchText, StringComparison.InvariantCultureIgnoreCase) >= 0);
            }
            else if (!string.IsNullOrWhiteSpace(ProductCodeSearchText))
            {
                filteredList = PendingBloodUnitFullList.Where(x => x.PendingBloodUnitTestModel.BloodUnitModel != null
                                                                && x.PendingBloodUnitTestModel.BloodUnitModel.ProductCode.IndexOf(ProductCodeSearchText, StringComparison.InvariantCultureIgnoreCase) >= 0);
            }
            else
            {
                filteredList = PendingBloodUnitFullList.ToList();
            }

            // BR_115.11 - Order by Unit ID 
            PendingBloodUnitList = filteredList.OrderBy(x => x.PendingBloodUnitTestModel.BloodUnitId).ToObservableCollection();
        }

        /// <summary>
        /// Handles checking/unchecking a given test
        /// </summary>
        /// <param name="selectableTest"></param>
        public void OnItemSelectedCommand(ISelectableTestViewModel selectableTest)
        {
            base.ToggleCheckBoxesEnabledState(selectableTest, PendingBloodUnitList);

            var selectedUnit = selectableTest as BloodUnitItemViewModel;

            if (selectedUnit == null)
            {
                return;
            }            

            // BR_115.08 - No Product code exists so we need to prompt the user to select one
            if (selectedUnit != null && selectedUnit.BloodUnitModel == null)
            {
                PromptForBloodUnit(selectedUnit);

                if (selectedUnit.BloodUnitModel == null)
                {
                    UncheckSelectedTest(selectableTest);
                }
            }

            if (selectedUnit != null && selectableTest.IsSelected)
            {
                selectedUnit.PromptUserWithWarnings(VbecsBaseForm);

                selectedUnit.PromptForExceptionOverrides(VbecsBaseForm);

                if (!selectedUnit.AreAllExceptionOverridesProcessed())
                {
                    UncheckSelectedTest(selectableTest);
                }
                else
                {
                    if (!VbecsBaseForm.SetLocks(selectedUnit.BloodUnitModel.BloodUnitStatusGuid))
                    {
                        UncheckSelectedTest(selectableTest);
                    }
                }
            }
            else
            {
                if (selectedUnit != null && selectedUnit.BloodUnitModel != null) VbecsBaseForm.ReleaseLocks(selectedUnit.BloodUnitModel.BloodUnitStatusGuid);
            }
        }

        /// <summary>
        /// Uncheck the given test when the UI threads available again via Dispatcher.BeginInvoke 
        /// </summary>
        /// <param name="selectableTest"></param>
        private void UncheckSelectedTest(ISelectableTestViewModel selectableTest)
        {
            selectableTest.IsSelected = false;

            var selectedUnit = selectableTest as BloodUnitItemViewModel;
            if (selectedUnit != null &&
                selectedUnit.BloodUnitModel != null)
            {
                VbecsBaseForm.ReleaseLocks(selectedUnit.BloodUnitModel.BloodUnitStatusGuid);
            }

            // No product code was selected so we will uncheck the user's selection
            System.Windows.Application.Current.Dispatcher.BeginInvoke(new System.Action(() =>
            {
                selectableTest.IsSelected = false;
                base.ToggleCheckBoxesEnabledState(selectableTest, PendingBloodUnitList);
            }));
        }

        /// <summary>
        /// OnProductCodeCommand
        /// </summary>
        /// <param name="selectedUnit"></param>
        private void OnProductCodeCommand(BloodUnitItemViewModel selectedUnit)
        {
            PromptForBloodUnit(selectedUnit);
        }

        /// <summary>
        /// Prompt the user to enter a product code
        /// </summary>
        /// <param name="selectedUnit"></param>
        private void PromptForBloodUnit(BloodUnitItemViewModel selectedUnit)
        {
            if (selectedUnit != null)
            {
                var originalTestValidationStatus = TestValidationStatus.Unknown;
                if (selectedUnit.TestValidation != null)
                {
                    originalTestValidationStatus = selectedUnit.TestValidation.TestValidationStatus;
                }
                var dlg = ViewConstructor.CreateFormAndView<BloodUnitProductCodeView, FrmWPFContainer>();
                var dlgViewModel = new BloodUnitProductCodeViewModel(selectedUnit.PendingBloodUnitTestModel.BloodUnitId, selectedUnit.BloodUnitModel, dlg);
                dlgViewModel.LogoHeaderText = "Select Product Code";
                dlg.WPFUserControl.DataContext = dlgViewModel;
                var dlgResult = dlg.ShowDialogFromLockedParent(VbecsBaseForm, false);

                if (dlgResult == DialogResult.OK &&
                    dlgViewModel.SelectedBloodUnit != null)
                {
                    selectedUnit.BloodUnitModel = dlgViewModel.SelectedBloodUnit.Clone();

                    if (selectedUnit.TestValidation == null ||
                        originalTestValidationStatus != selectedUnit.TestValidation.TestValidationStatus)
                    {
                        // Test Validation changed so uncheck all other tests
                        // and select this test
                        base.UncheckAll(PendingBloodUnitList);
                        foreach (BloodUnitItemViewModel tmp in PendingBloodUnitList)
                        {
                            if (tmp.BloodUnitModel != null) VbecsBaseForm.ReleaseLocks(tmp.BloodUnitModel.BloodUnitStatusGuid);
                        }

                        System.Windows.Application.Current.Dispatcher.BeginInvoke(new System.Action(() =>
                        {
                            selectedUnit.IsSelected = true;
                            OnItemSelectedCommand(selectedUnit);
                        }));
                    }
                }
            }
        }

        /// <summary>
        /// Check for unsaved changes
        /// </summary>
        /// <param name="other"></param>
        /// <returns></returns>
        public bool IsDirty()
        {
            foreach (var testViewModel in PendingBloodUnitFullList)
            {
                if (testViewModel.IsDirty())
                {
                    return true;
                }
            }

            return false;
        }
    }
}
